# -*- coding: utf-8 -*-
"""葡月ERP - 库存路由"""

from flask import Blueprint, render_template, request, redirect, url_for
from datetime import datetime
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

inventory_bp = Blueprint('inventory', __name__)


@inventory_bp.route('/inventory')
def inventory():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    products = inv.get("products", [])
    sales = load('sales', {"orders": []})
    
    # 基础库存统计
    total_value = sum(p.get("quantity", 0) * p.get("cost", 0) for p in products)
    total_quantity = sum(p.get("quantity", 0) for p in products)
    total_bottles = sum(p.get("quantity", 0) * (6 if "6瓶" in str(p.get("spec", "")) else 4) for p in products)
    low_stock = len([p for p in products if p.get("min_stock", 0) and p.get("quantity", 0) < p.get("min_stock", 0)])
    
    # 计算销售数据用于周转分析
    sales_orders = sales.get("orders", [])
    product_sales = {}
    for o in sales_orders:
        pname = o.get("product", "")
        if pname not in product_sales:
            product_sales[pname] = 0
        product_sales[pname] += o.get("quantity", 0)
    
    # 库存周转分析
    turnover_data = []
    for p in products:
        pname = p.get("name", "")
        qty = p.get("quantity", 0)
        sold = product_sales.get(pname, 0)
        
        # 日均销量（假设销售30天）
        daily_sales = sold / 30 if sold > 0 else 0
        # 可销售天数
        days_left = int(qty / daily_sales) if daily_sales > 0 else 999
        
        # 库存状态
        if days_left == 999:
            status = "无销售"
        elif days_left < 7:
            status = "紧急"
        elif days_left < 14:
            status = "预警"
        elif days_left > 60:
            status = "滞销"
        elif days_left > 30:
            status = "充足"
        else:
            status = "正常"
        
        # 建议补货量
        suggest = 0
        if daily_sales > 0:
            target_days = 30  # 目标库存天数
            target_qty = daily_sales * target_days
            suggest = max(0, int(target_qty - qty))
        
        turnover_data.append({
            "name": pname,
            "quantity": qty,
            "sold": sold,
            "daily_sales": round(daily_sales, 2),
            "days_left": days_left,
            "status": status,
            "suggest": suggest,
            "value": qty * p.get("cost", 0)
        })
    
    # 按状态分组统计
    status_count = {"紧急": 0, "预警": 0, "正常": 0, "充足": 0, "滞销": 0, "无销售": 0}
    for t in turnover_data:
        status_count[t["status"]] = status_count.get(t["status"], 0) + 1
    
    # 库存价值分布
    value_by_product = sorted(turnover_data, key=lambda x: x["value"], reverse=True)
    
    # 预警商品列表
    warnings = []
    for p in products:
        qty = p.get("quantity", 0)
        min_qty = p.get("min_stock", 0)
        if min_qty and qty < min_qty:
            warnings.append({
                "name": p.get("name", ""),
                "quantity": qty,
                "min_stock": min_qty,
                "suggest": min_qty * 2 - qty
            })
    
    inventory_stats = {
        "total_value": total_value,
        "total_quantity": total_quantity,
        "total_bottles": total_bottles,
        "total_products": len(products),
        "low_stock": low_stock,
        "status_count": status_count,
        "total_sold": sum(product_sales.values())
    }
    
    return render_with_user('index.html', sessions, page='inventory',
        products=products,
        inventory_stats=inventory_stats,
        turnover=turnover_data,
        warnings=warnings,
        value_by_product=value_by_product[:5])


@inventory_bp.route('/products')
def products():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    products = inv.get("products", [])
    
    # 库存统计
    total_value = sum(p.get("quantity", 0) * p.get("cost", 0) for p in products)
    total_quantity = sum(p.get("quantity", 0) for p in products)
    low_stock = len([p for p in products if p.get("min_stock", 0) and p.get("quantity", 0) < p.get("min_stock", 0)])
    
    inventory_stats = {
        "total_value": total_value,
        "total_quantity": total_quantity,
        "total_products": len(products),
        "low_stock": low_stock
    }
    
    product_stats = {
        "total_products": len(products),
        "total_value": total_value,
        "low_stock": low_stock
    }
    
    return render_with_user('index.html', sessions, page='products',
        products=products,
        inventory_stats=inventory_stats,
        product_stats=product_stats)


@inventory_bp.route('/add_product', methods=['POST'])
def add_product():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    
    product = {
        "id": len(inv.get("products", [])) + 1,
        "name": request.form.get('name', '').strip(),
        "category": request.form.get('category', '').strip(),
        "quantity": int(request.form.get('quantity', 0)),
        "unit": request.form.get('unit', '瓶'),
        "cost_price": float(request.form.get('cost_price', 0)),
        "selling_price": float(request.form.get('selling_price', 0)),
        "min_stock": int(request.form.get('min_stock', 0)),
        "supplier": request.form.get('supplier', '').strip(),
        "date": datetime.now().strftime("%Y-%m-%d")
    }
    
    inv.setdefault("products", []).append(product)
    save('inventory', inv)
    
    return redirect(url_for('inventory.products'))


@inventory_bp.route('/edit_product/<int:product_id>', methods=['GET', 'POST'])
def edit_product(product_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    
    if request.method == 'POST':
        for p in inv.get("products", []):
            if p.get("id") == product_id:
                p["name"] = request.form.get('name', '').strip()
                p["category"] = request.form.get('category', '').strip()
                p["quantity"] = int(request.form.get('quantity', 0))
                p["unit"] = request.form.get('unit', '瓶')
                p["cost_price"] = float(request.form.get('cost_price', 0))
                p["selling_price"] = float(request.form.get('selling_price', 0))
                p["min_stock"] = int(request.form.get('min_stock', 0))
                p["supplier"] = request.form.get('supplier', '').strip()
                break
        save('inventory', inv)
        return redirect(url_for('inventory.products'))
    
    product = None
    for p in inv.get("products", []):
        if p.get("id") == product_id:
            product = p
            break
    
    return render_with_user('index.html', sessions, page='edit_product', product=product)


@inventory_bp.route('/delete_product/<int:product_id>')
def delete_product(product_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    inv["products"] = [p for p in inv.get("products", []) if p.get("id") != product_id]
    save('inventory', inv)
    
    return redirect(url_for('inventory.products'))


@inventory_bp.route('/stock_check')
def stock_check():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    check_data = load('stock_check', {"checks":[]})
    
    return render_with_user('index.html', sessions, page='stock_check',
        products=inv.get("products", []),
        checks=check_data.get("checks", []))


@inventory_bp.route('/add_stock_check', methods=['POST'])
def add_stock_check():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    check_data = load('stock_check', {"checks":[]})
    
    check = {
        "id": len(check_data.get("checks", [])) + 1,
        "product": request.form.get('product', '').strip(),
        "actual_quantity": int(request.form.get('actual_quantity', 0)),
        "system_quantity": int(request.form.get('system_quantity', 0)),
        "difference": int(request.form.get('actual_quantity', 0)) - int(request.form.get('system_quantity', 0)),
        "operator": user.get('name', ''),
        "date": datetime.now().strftime("%Y-%m-%d"),
        "remark": request.form.get('remark', '').strip()
    }
    
    check_data.setdefault("checks", []).append(check)
    save('stock_check', check_data)
    
    return redirect(url_for('inventory.stock_check'))


# ========== 库存调整 ==========
@inventory_bp.route('/adjust_stock/<int:product_id>', methods=['POST'])
def adjust_stock(product_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    adjustment = int(request.form.get('adjustment', 0))
    reason = request.form.get('reason', '').strip()
    
    inv = load('inventory', {"products": []})
    
    for p in inv.get("products", []):
        if p.get("id") == product_id:
            old_qty = p.get("quantity", 0)
            new_qty = old_qty + adjustment
            if new_qty < 0:
                new_qty = 0
            p["quantity"] = new_qty
            p["last_adjustment"] = {
                "date": datetime.now().strftime("%Y-%m-%d"),
                "adjustment": adjustment,
                "reason": reason,
                "operator": user.get("name", "")
            }
            break
    
    save('inventory', inv)
    return redirect(url_for('inventory.inventory'))


# ========== 库存预警 ==========
@inventory_bp.route('/inventory_warning')
def inventory_warning():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    products = inv.get("products", [])
    
    # 库存预警
    warnings = []
    low_stock = []
    out_of_stock = []
    
    for p in products:
        qty = p.get("quantity", 0)
        min_qty = p.get("min_stock", 0)
        
        if qty == 0:
            out_of_stock.append(p)
        elif min_qty and qty < min_qty:
            low_stock.append({
                "name": p.get("name"),
                "quantity": qty,
                "min_stock": min_qty,
                "shortage": min_qty - qty
            })
    
    warning_stats = {
        "out_of_stock": len(out_of_stock),
        "low_stock": len(low_stock),
        "total_warnings": len(out_of_stock) + len(low_stock)
    }
    
    return render_with_user('index.html', sessions, page='inventory_warning',
        warnings=warnings,
        low_stock=low_stock,
        out_of_stock=out_of_stock,
        warning_stats=warning_stats)
