# -*- coding: utf-8 -*-
"""葡月ERP - 客户和用户路由"""

from flask import Blueprint, render_template, request, redirect, url_for
from datetime import datetime
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

customers_bp = Blueprint('customers', __name__)
users_bp = Blueprint('users', __name__)


# ========== 客户管理 ==========

@customers_bp.route('/customers')
def customers():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    cust = load('customers', {"list":[]})
    customer_list = cust.get("list", [])
    sales = load('sales', {"orders": []})
    
    # 客户消费统计
    customer_sales = {}
    for o in sales.get("orders", []):
        cname = o.get("customer", "")
        if cname not in customer_sales:
            customer_sales[cname] = {"orders": 0, "total": 0, "profit": 0}
        customer_sales[cname]["orders"] += 1
        customer_sales[cname]["total"] += o.get("total_amount", 0)
        customer_sales[cname]["profit"] += o.get("profit", 0)
    
    # 更新客户消费数据
    for c in customer_list:
        cname = c.get("name", "")
        if cname in customer_sales:
            c["order_count"] = customer_sales[cname]["orders"]
            c["total_sales"] = customer_sales[cname]["total"]
            c["total_profit"] = customer_sales[cname]["profit"]
        else:
            c["order_count"] = 0
            c["total_sales"] = 0
            c["total_profit"] = 0
    
    # 客户等级统计
    vip_count = len([c for c in customer_list if c.get("level") == "VIP"])
    normal_count = len([c for c in customer_list if c.get("level") == "普通"])
    silver_count = len([c for c in customer_list if c.get("level") == "银卡"])
    gold_count = len([c for c in customer_list if c.get("level") == "金卡"])
    
    # 客户统计
    customer_stats = {
        "total": len(customer_list),
        "vip": vip_count,
        "gold": gold_count,
        "silver": silver_count,
        "normal": normal_count,
        "total_sales": sum(c.get("total_sales", 0) for c in customer_list)
    }
    
    save('customers', cust)
    
    return render_with_user('index.html', sessions, page='customers',
        customers=customer_list,
        customer_stats=customer_stats)


@customers_bp.route('/add_customer', methods=['POST'])
def add_customer():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    cust = load('customers', {"list":[]})
    
    customer = {
        "id": len(cust.get("list", [])) + 1,
        "name": request.form.get('name', '').strip(),
        "phone": request.form.get('phone', '').strip(),
        "address": request.form.get('address', '').strip(),
        "level": request.form.get('level', '普通'),
        "remark": request.form.get('remark', '').strip(),
        "date": datetime.now().strftime("%Y-%m-%d")
    }
    
    cust.setdefault("list", []).append(customer)
    save('customers', cust)
    
    return redirect(url_for('customers.customers'))


@customers_bp.route('/edit_customer/<int:customer_id>', methods=['GET', 'POST'])
def edit_customer(customer_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    cust = load('customers', {"list":[]})
    
    if request.method == 'POST':
        for c in cust.get("list", []):
            if c.get("id") == customer_id:
                c["name"] = request.form.get('name', '').strip()
                c["phone"] = request.form.get('phone', '').strip()
                c["address"] = request.form.get('address', '').strip()
                c["level"] = request.form.get('level', '普通')
                c["remark"] = request.form.get('remark', '').strip()
                break
        save('customers', cust)
        return redirect(url_for('customers.customers'))
    
    customer = None
    for c in cust.get("list", []):
        if c.get("id") == customer_id:
            customer = c
            break
    
    return render_with_user('index.html', sessions, page='edit_customer', customer=customer)


@customers_bp.route('/delete_customer/<int:customer_id>')
def delete_customer(customer_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    cust = load('customers', {"list":[]})
    cust["list"] = [c for c in cust.get("list", []) if c.get("id") != customer_id]
    save('customers', cust)
    
    return redirect(url_for('customers.customers'))


# ========== 用户管理 ==========

@users_bp.route('/users')
def users_page():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get("role") != "admin":
        return "无权限访问", 403
    
    users = load('users', {"users": []})
    return render_with_user('index.html', sessions, page='users', users=users.get("users", []))


@users_bp.route('/add_user', methods=['POST'])
def add_user():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get("role") != "admin":
        return "无权限访问", 403
    
    users = load('users', {"users": []})
    
    new_user = {
        "id": len(users.get("users", [])) + 1,
        "username": request.form.get('username', '').strip(),
        "password": request.form.get('password', '').strip(),
        "name": request.form.get('name', '').strip(),
        "role": request.form.get('role', 'user'),
        "permissions": request.form.get('permissions', '').split(',') if request.form.get('permissions') else [],
        "wecom_user": request.form.get('wecom_user', '').strip(),
        "date": datetime.now().strftime("%Y-%m-%d")
    }
    
    users.setdefault("users", []).append(new_user)
    save('users', users)
    
    return redirect(url_for('users.users_page'))


@users_bp.route('/edit_user/<int:uid>', methods=['GET', 'POST'])
def edit_user(uid):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get("role") != "admin":
        return "无权限访问", 403
    
    users = load('users', {"users": []})
    
    if request.method == 'POST':
        for u in users.get("users", []):
            if u.get("id") == uid:
                u["username"] = request.form.get('username', '').strip()
                if request.form.get('password'):
                    u["password"] = request.form.get('password', '').strip()
                u["name"] = request.form.get('name', '').strip()
                u["role"] = request.form.get('role', 'user')
                u["permissions"] = request.form.get('permissions', '').split(',') if request.form.get('permissions') else []
                u["wecom_user"] = request.form.get('wecom_user', '').strip()
                break
        save('users', users)
        return redirect(url_for('users.users_page'))
    
    edit_user = None
    for u in users.get("users", []):
        if u.get("id") == uid:
            edit_user = u
            break
    
    return render_with_user('index.html', sessions, page='edit_user', edit_user=edit_user)


@users_bp.route('/delete_user/<int:uid>')
def delete_user(uid):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get("role") != "admin":
        return "无权限访问", 403
    
    users = load('users', {"users": []})
    users["users"] = [u for u in users.get("users", []) if u.get("id") != uid]
    save('users', users)
    
    return redirect(url_for('users.users_page'))


@users_bp.route('/change_password', methods=['POST'])
def change_password():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    old_password = request.form.get('old_password', '').strip()
    new_password = request.form.get('new_password', '').strip()
    
    users = load('users', {"users": []})
    
    for u in users.get("users", []):
        if u.get("id") == user.get("user_id"):
            if u.get("password") == old_password:
                u["password"] = new_password
                save('users', users)
                return "密码修改成功"
            return "原密码错误"
    
    return "用户不存在"
