# -*- coding: utf-8 -*-
"""葡月ERP - 费用报销路由"""

from flask import Blueprint, render_template, request, redirect, url_for
from datetime import datetime
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

expenses_bp = Blueprint('expenses', __name__)


@expenses_bp.route('/expenses')
def expenses():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    expense_list = expenses_data.get("list", [])
    
    # 基础统计
    total = sum(e.get("amount", 0) for e in expense_list)
    pending = sum(e.get("amount", 0) for e in expense_list if e.get("status") == "待审批")
    approved = sum(e.get("amount", 0) for e in expense_list if e.get("status") == "已批准")
    paid = sum(e.get("amount", 0) for e in expense_list if e.get("status") == "已付款")
    rejected = sum(e.get("amount", 0) for e in expense_list if e.get("status") == "已拒绝")
    
    # 按类型统计
    type_stats = {}
    for e in expense_list:
        t = e.get("type", "其他")
        if t not in type_stats:
            type_stats[t] = {"count": 0, "amount": 0}
        type_stats[t]["count"] += 1
        type_stats[t]["amount"] += e.get("amount", 0)
    
    # 按月份统计
    monthly_stats = {}
    for e in expense_list:
        month = e.get("date", "")[:7]  # YYYY-MM
        if month not in monthly_stats:
            monthly_stats[month] = {"count": 0, "amount": 0}
        monthly_stats[month]["count"] += 1
        monthly_stats[month]["amount"] += e.get("amount", 0)
    
    # 排序
    type_list = sorted([{"type": k, "count": v["count"], "amount": v["amount"]} for k, v in type_stats.items()], key=lambda x: x["amount"], reverse=True)
    monthly_list = sorted([{"month": k, "count": v["count"], "amount": v["amount"]} for k, v in monthly_stats.items()], key=lambda x: x["month"], reverse=True)
    
    # 状态统计
    status_counts = {
        "pending": len([e for e in expense_list if e.get("status") == "待审批"]),
        "approved": len([e for e in expense_list if e.get("status") == "已批准"]),
        "paid": len([e for e in expense_list if e.get("status") == "已付款"]),
        "rejected": len([e for e in expense_list if e.get("status") == "已拒绝"])
    }
    
    expense_stats = {
        "total": total,
        "pending": pending,
        "approved": approved,
        "paid": paid,
        "rejected": rejected,
        "count": len(expense_list),
        "avg_amount": total / len(expense_list) if expense_list else 0,
        "status_counts": status_counts,
        "type_list": type_list,
        "monthly_list": monthly_list[:6]  # 最近6个月
    }
    
    return render_with_user('index.html', sessions, page='expenses',
        expenses=expense_list,
        expense_stats=expense_stats,
        top_expense_types=type_list[:5])


@expenses_bp.route('/add_expense', methods=['POST'])
def add_expense():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    expense = {
        "id": len(expenses_data.get("list", [])) + 1,
        "type": request.form.get('type', '').strip(),
        "amount": float(request.form.get('amount', 0)),
        "description": request.form.get('description', '').strip(),
        "applicant": user.get('name', ''),
        "status": "待审批",
        "date": datetime.now().strftime("%Y-%m-%d")
    }
    
    expenses_data.setdefault("list", []).append(expense)
    save('expenses', expenses_data)
    
    return redirect(url_for('expenses.expenses'))


@expenses_bp.route('/pay_expense/<int:expense_id>')
def pay_expense(expense_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    for e in expenses_data.get("list", []):
        if e.get("id") == expense_id:
            e["status"] = "已付款"
            e["pay_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('expenses', expenses_data)
    return redirect(url_for('expenses.expenses'))


@expenses_bp.route('/update_expense_status/<int:expense_id>/<status>')
def update_expense_status(expense_id, status):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    for e in expenses_data.get("list", []):
        if e.get("id") == expense_id:
            e["status"] = status
            e["approve_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('expenses', expenses_data)
    return redirect(url_for('expenses.expenses'))


@expenses_bp.route('/delete_expense/<int:expense_id>')
def delete_expense(expense_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    expenses_data["list"] = [e for e in expenses_data.get("list", []) if e.get("id") != expense_id]
    save('expenses', expenses_data)
    
    return redirect(url_for('expenses.expenses'))
