# -*- coding: utf-8 -*-
"""葡月ERP - 工具函数"""

import json
import os
from flask import request, current_app


def load(key, default=None):
    """加载JSON数据"""
    if default is None:
        default = {}
    filepath = os.path.join(current_app.config['DATA_DIR'], current_app.config['FILES'].get(key, f'{key}.json'))
    os.makedirs(os.path.dirname(filepath), exist_ok=True)
    try:
        with open(filepath, 'r', encoding='utf-8') as f:
            return json.load(f)
    except (FileNotFoundError, json.JSONDecodeError):
        return default


def save(key, data):
    """保存JSON数据"""
    filepath = os.path.join(current_app.config['DATA_DIR'], current_app.config['FILES'].get(key, f'{key}.json'))
    os.makedirs(os.path.dirname(filepath), exist_ok=True)
    with open(filepath, 'w', encoding='utf-8') as f:
        json.dump(data, f, ensure_ascii=False, indent=2)
    return True


def get_current_user(sessions):
    """获取当前用户"""
    # 企业微信用户
    wecom_user = request.args.get('wecom_user') or request.headers.get('X-Wecom-User')
    if wecom_user:
        users = load('users', {"users": []})
        admin_whitelist = load('users', {"admin_whitelist": []}).get("admin_whitelist", [])
        for u in users.get("users", []):
            if u.get("wecom_user") == wecom_user:
                role = u.get("role", "user")
                if wecom_user in admin_whitelist:
                    role = "admin"
                return {
                    "username": u.get("username", wecom_user),
                    "name": u.get("name", wecom_user),
                    "role": role,
                    "user_id": u.get("id"),
                    "permissions": u.get("permissions", []),
                    "is_wecom": True
                }
        return {
            "username": wecom_user,
            "name": wecom_user,
            "role": "user",
            "is_wecom": True
        }
    
    # Session 登录
    session_id = request.cookies.get('session_id')
    if session_id and session_id in sessions:
        user = sessions[session_id]
        users = load('users', {"users": []})
        admin_whitelist = load('users', {"admin_whitelist": []}).get("admin_whitelist", [])
        for u in users.get("users", []):
            if u.get("id") == user.get("user_id"):
                user['permissions'] = u.get("permissions")
                if u.get("wecom_user") in admin_whitelist:
                    user['role'] = "admin"
                break
        return user
    
    # 默认返回管理员用户，无需登录
    return {
        "username": "admin",
        "name": "管理员",
        "role": "admin",
        "user_id": 1,
        "permissions": ["sales", "purchase", "inventory", "customers", "financial", "expenses", "backup", "users"]
    }


def check_permission(user, page):
    """检查权限"""
    if not user:
        return False
    
    admin_only = ["users", "backup", "restore", "add_user", "edit_user", "delete_user", "system_settings"]
    if page in admin_only:
        return user.get("role") == "admin"
    
    perms = user.get("permissions")
    if perms:
        return page in perms
    
    role = user.get("role", "guest")
    return page in current_app.config['ROLE_PERMISSIONS'].get(role, [])


def render_with_user(template, sessions, **kwargs):
    """带用户信息渲染模板"""
    from flask import render_template
    user = get_current_user(sessions)
    kwargs['current_user'] = user
    kwargs['user_permissions'] = user.get("permissions") or current_app.config['ROLE_PERMISSIONS'].get(user.get("role", "guest"), [])
    return render_template(template, **kwargs)
