# -*- coding: utf-8 -*-
"""葡月ERP - 财务管理路由"""

import json
import os
from flask import Blueprint, render_template, request, redirect, url_for, send_file, current_app
from datetime import datetime, timedelta
from collections import defaultdict
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

financial_bp = Blueprint('financial', __name__)


@financial_bp.route('/financial')
def financial():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders": []})
    purchase = load('purchase', {"orders": []})
    expenses = load('expenses', {"list": []})
    inventory = load('inventory', {"products": []})
    transactions = load('transactions', {"list": []})
    
    # 销售数据
    orders = sales.get("orders", [])
    total_sales = sum(o.get("total_amount", 0) for o in orders)
    total_profit = sum(o.get("profit", 0) for o in orders)
    # 包含部分收款
    received = sum(o.get("paid_amount", 0) for o in orders if o.get("payment_status") in ["已收款", "部分收款"])
    receivable = total_sales - received
    
    # 采购数据
    purchase_orders = purchase.get("orders", [])
    total_purchase = sum(o.get("total_amount", 0) for o in purchase_orders)
    # 包含部分付款
    paid = sum(o.get("paid_amount", 0) for o in purchase_orders if o.get("payment_status") in ["已付款", "部分付款"])
    payable = total_purchase - paid
    
    # 费用数据
    expense_list = expenses.get("list", [])
    total_expenses = sum(e.get("amount", 0) for e in expense_list if e.get("status") == "已付款")
    
    # 库存价值
    stock_value = sum(p.get("quantity", 0) * p.get("cost_price", 0) for p in inventory.get("products", []))
    
    # 资金流向前 (回款 - 付款 - 费用)
    cash_balance = received - paid - total_expenses
    
    # 收集所有交易流水并按账户统计
    all_transactions = []
    account_balances = {"对公": 0, "现金": 0}
    
    # 销售收款记录
    for o in orders:
        if "payment_records" in o:
            for r in o.get("payment_records", []):
                account = r.get("account", "对公")
                amount = r.get("amount", 0)
                all_transactions.append({
                    "date": r.get("date", ""),
                    "type": "销售收款",
                    "account": account,
                    "amount": amount,
                    "ref": f"订单#{o.get('id')}",
                    "customer": o.get("customer", "")
                })
                if account in account_balances:
                    account_balances[account] += amount
    
    # 采购付款记录
    for o in purchase_orders:
        if "payment_records" in o:
            for r in o.get("payment_records", []):
                account = r.get("account", "对公")
                amount = r.get("amount", 0)
                all_transactions.append({
                    "date": r.get("date", ""),
                    "type": "采购付款",
                    "account": account,
                    "amount": -amount,
                    "ref": f"采购#{o.get('id')}",
                    "customer": o.get("supplier", "")
                })
                if account in account_balances:
                    account_balances[account] -= amount
    
    # 费用支出记录
    for e in expense_list:
        if e.get("status") == "已付款":
            account = e.get("pay_account", "现金")
            amount = e.get("amount", 0)
            all_transactions.append({
                "date": e.get("pay_date", e.get("date", "")),
                "type": "费用支出",
                "account": account,
                "amount": -amount,
                "ref": f"费用#{e.get('id')}",
                "customer": e.get("description", "")
            })
            if account in account_balances:
                account_balances[account] -= amount
    
    # 按日期排序
    all_transactions.sort(key=lambda x: x["date"], reverse=True)
    
    # 统计
    financial_stats = {
        "total_sales": total_sales,
        "total_profit": total_profit,
        "total_purchase": total_purchase,
        "total_expenses": total_expenses,
        "net_profit": total_profit - total_expenses,
        "sales_count": len(orders),
        "purchase_count": len(purchase_orders),
        "received": received,
        "receivable": receivable,
        "paid": paid,
        "payable": payable,
        "stock_value": stock_value,
        "cash_balance": cash_balance,
        "corporate_balance": account_balances.get("对公", 0),
        "cash_account_balance": account_balances.get("现金", 0)
    }
    
    # 加载配置
    config_path = os.path.join(current_app.config['DATA_DIR'], 'config.json')
    try:
        with open(config_path, 'r', encoding='utf-8') as f:
            config_data = json.load(f)
            registered_capital = config_data.get('registered_capital', 1000000)
    except:
        registered_capital = 1000000

    # 模板需要的格式
    financial = {
        "sales": total_sales,
        "cost": sum(o.get("quantity", 0) * o.get("cost_price", 0) for o in orders),
        "profit": total_profit,
        "margin": round(total_profit / total_sales * 100, 1) if total_sales else 0,
        "purchase": total_purchase,
        "stock_value": stock_value,
        "receivable": receivable,
        "payable": payable,
        "cash": cash_balance,
        "corporate": received,
        "capital": registered_capital
    }
    
    # 销售月度报表
    monthly_sales = defaultdict(lambda: {"orders": 0, "quantity": 0, "sales": 0, "profit": 0})
    for o in orders:
        month = o.get("date", "")[:7]  # YYYY-MM
        monthly_sales[month]["orders"] += 1
        monthly_sales[month]["quantity"] += o.get("quantity", 0)
        monthly_sales[month]["sales"] += o.get("total_amount", 0)
        monthly_sales[month]["profit"] += o.get("profit", 0)
    
    monthly_report = sorted([{
        "month": k,
        "orders": v["orders"],
        "quantity": v["quantity"],
        "sales": v["sales"],
        "profit": v["profit"]
    } for k, v in monthly_sales.items()], key=lambda x: x["month"], reverse=True)[:12]
    
    # 商品利润报表
    product_profit = defaultdict(lambda: {"quantity": 0, "sales": 0, "profit": 0})
    for o in orders:
        pname = o.get("product", "")
        product_profit[pname]["quantity"] += o.get("quantity", 0)
        product_profit[pname]["sales"] += o.get("total_amount", 0)
        product_profit[pname]["profit"] += o.get("profit", 0)
    
    product_report = sorted([{
        "name": k[:25],
        "quantity": v["quantity"],
        "sales": v["sales"],
        "profit": v["profit"]
    } for k, v in product_profit.items()], key=lambda x: x["profit"], reverse=True)
    
    # 客户分析报表
    customer_profit = defaultdict(lambda: {"orders": 0, "quantity": 0, "sales": 0, "profit": 0})
    for o in orders:
        cname = o.get("customer", "")
        customer_profit[cname]["orders"] += 1
        customer_profit[cname]["quantity"] += o.get("quantity", 0)
        customer_profit[cname]["sales"] += o.get("total_amount", 0)
        customer_profit[cname]["profit"] += o.get("profit", 0)
    
    customer_report = sorted([{
        "name": k[:20],
        "orders": v["orders"],
        "quantity": v["quantity"],
        "sales": v["sales"],
        "profit": v["profit"]
    } for k, v in customer_profit.items()], key=lambda x: x["sales"], reverse=True)
    
    return render_with_user('index.html', sessions, page='financial',
        financial_stats=financial_stats,
        financial=financial,
        orders=orders,
        purchase_orders=purchase_orders,
        transactions=all_transactions,
        monthly_report=monthly_report,
        product_report=product_report,
        customer_report=customer_report,
        account_balances=account_balances)


@financial_bp.route('/receive_payment/<int:order_id>')
def receive_payment(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders": []})
    
    for o in sales.get("orders", []):
        if o.get("id") == order_id:
            o["payment_status"] = "已收款"
            o["payment_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('sales', sales)
    return redirect(url_for('financial.financial'))


@financial_bp.route('/payment_history/<int:order_id>')
def payment_history(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders": []})
    
    order = None
    for o in sales.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='payment_history', order=order)
