# -*- coding: utf-8 -*-
"""葡月ERP - 费用报销路由"""

from flask import Blueprint, render_template, request, redirect, url_for
from datetime import datetime
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

expenses_bp = Blueprint('expenses', __name__)


@expenses_bp.route('/expenses')
def expenses():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    expense_list = expenses_data.get("list", [])
    total = sum(e.get("amount", 0) for e in expense_list)
    pending = sum(e.get("amount", 0) for e in expense_list if e.get("status") == "待审批")
    approved = sum(e.get("amount", 0) for e in expense_list if e.get("status") == "已批准")
    
    expense_stats = {
        "total": total,
        "pending": pending,
        "approved": approved,
        "count": len(expense_list),
        "paid": sum(e.get("amount", 0) for e in expense_list if e.get("status") == "已付款"),
        "rejected": 0,
        "status_counts": {
            "pending": pending,
            "approved": approved,
            "paid": sum(e.get("amount", 0) for e in expense_list if e.get("status") == "已付款"),
            "rejected": 0
        }
    }
    
    return render_with_user('index.html', sessions, page='expenses',
        expenses=expense_list,
        expense_stats=expense_stats)


@expenses_bp.route('/add_expense', methods=['POST'])
def add_expense():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    expense = {
        "id": len(expenses_data.get("list", [])) + 1,
        "type": request.form.get('type', '').strip(),
        "amount": float(request.form.get('amount', 0)),
        "description": request.form.get('description', '').strip(),
        "applicant": user.get('name', ''),
        "status": "待审批",
        "date": datetime.now().strftime("%Y-%m-%d")
    }
    
    expenses_data.setdefault("list", []).append(expense)
    save('expenses', expenses_data)
    
    return redirect(url_for('expenses.expenses'))


@expenses_bp.route('/pay_expense/<int:expense_id>')
def pay_expense(expense_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    for e in expenses_data.get("list", []):
        if e.get("id") == expense_id:
            e["status"] = "已付款"
            e["pay_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('expenses', expenses_data)
    return redirect(url_for('expenses.expenses'))


@expenses_bp.route('/update_expense_status/<int:expense_id>/<status>')
def update_expense_status(expense_id, status):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    
    for e in expenses_data.get("list", []):
        if e.get("id") == expense_id:
            e["status"] = status
            e["approve_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('expenses', expenses_data)
    return redirect(url_for('expenses.expenses'))


@expenses_bp.route('/delete_expense/<int:expense_id>')
def delete_expense(expense_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    expenses_data = load('expenses', {"list": []})
    expenses_data["list"] = [e for e in expenses_data.get("list", []) if e.get("id") != expense_id]
    save('expenses', expenses_data)
    
    return redirect(url_for('expenses.expenses'))
