# -*- coding: utf-8 -*-
"""葡月ERP - 采购路由"""

from flask import Blueprint, render_template, request, redirect, url_for
from datetime import datetime
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

purchase_bp = Blueprint('purchase', __name__)


@purchase_bp.route('/purchase')
def purchase():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    products = load('inventory', {"products":[]}).get("products", [])
    orders = purchase_data.get("orders", [])
    
    # 采购统计
    total = sum(o.get("total_amount", 0) for o in orders)
    paid = sum(o.get("total_amount", 0) for o in orders if o.get("payment_status") == "已付款")
    pending = total - paid
    
    # 订单状态
    payment_status = {"已付款": 0, "待付款": 0}
    delivery_status = {"已收货": 0, "待收货": 0}
    for o in orders:
        if o.get("payment_status") == "已付款":
            payment_status["已付款"] += 1
        else:
            payment_status["待付款"] += 1
        
        if o.get("delivery_status") == "已收货":
            delivery_status["已收货"] += 1
        else:
            delivery_status["待收货"] += 1
    
    purchase_stats = {
        "total": total,
        "paid": paid,
        "pending": pending,
        "orders": len(orders),
        "quantity": sum(o.get("quantity", 0) for o in orders),
        "payment_status": payment_status,
        "delivery_status": delivery_status
    }
    
    return render_with_user('index.html', sessions, page='purchase',
        orders=orders,
        purchase_orders=orders,
        products=products,
        purchase_stats=purchase_stats)


@purchase_bp.route('/add_purchase', methods=['POST'])
def add_purchase():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    
    product = request.form.get('product', '').strip()
    quantity = int(request.form.get('quantity', 0))
    unit_price = float(request.form.get('unit_price', 0))
    supplier = request.form.get('supplier', '').strip()
    
    # 验证
    if not product or quantity <= 0 or unit_price < 0 or not supplier:
        return redirect(url_for('purchase.purchase'))
    
    total_amount = quantity * unit_price
    
    # 使用最大ID+1，避免重复
    orders = purchase_data.get("orders", [])
    max_id = max([o.get("id", 0) for o in orders]) if orders else 0
    
    order = {
        "id": max_id + 1,
        "product": product,
        "quantity": quantity,
        "unit_price": unit_price,
        "total_amount": total_amount,
        "supplier": supplier,
        "date": datetime.now().strftime("%Y-%m-%d"),
        "status": "待付款",
        "payment_status": "未付款",
        "delivery_status": "未收货",
        "paid_amount": 0
    }
    
    purchase_data.setdefault("orders", []).append(order)
    save('purchase', purchase_data)
    
    return redirect(url_for('purchase.purchase'))


@purchase_bp.route('/confirm_purchase_payment/<int:order_id>/<status>')
@purchase_bp.route('/confirm_purchase_payment/<int:order_id>/<status>/<account>')
@purchase_bp.route('/confirm_purchase_payment/<int:order_id>/<status>/<account>/<amount>')
def confirm_purchase_payment(order_id, status, account=None, amount=None):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            o["payment_status"] = status
            if account:
                o["payment_account"] = account
            if amount:
                o["paid_amount"] = float(amount)
            if status == "已付款":
                o["payment_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('purchase', purchase_data)
    return redirect(url_for('purchase.purchase'))


@purchase_bp.route('/pay_purchase/<int:order_id>')
def pay_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    products = load('inventory', {"products":[]}).get("products", [])
    
    order = None
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='pay_purchase',
        order=order, products=products)


@purchase_bp.route('/do_pay_purchase', methods=['POST'])
def do_pay_purchase():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    order_id = int(request.form.get('order_id', 0))
    account = request.form.get('account', '')
    amount = float(request.form.get('amount', 0))
    
    purchase_data = load('purchase', {"orders":[]})
    
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            # 获取之前的已付金额
            old_paid = o.get("paid_amount", 0)
            new_paid = old_paid + amount
            total = o.get("total_amount", 0)
            
            # 更新付款状态
            if new_paid >= total:
                o["payment_status"] = "已付款"
            elif new_paid > 0:
                o["payment_status"] = "部分付款"
            else:
                o["payment_status"] = "未付款"
            
            # 记录付款信息
            if "payment_records" not in o:
                o["payment_records"] = []
            
            o["payment_records"].append({
                "date": datetime.now().strftime("%Y-%m-%d %H:%M"),
                "account": account,
                "amount": amount,
                "operator": user.get("name", "")
            })
            
            o["paid_amount"] = new_paid
            o["payment_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('purchase', purchase_data)
    return redirect(url_for('purchase.purchase'))


@purchase_bp.route('/confirm_purchase_delivery/<int:order_id>/<status>')
def confirm_purchase_delivery(order_id, status):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    inv = load('inventory', {"products":[]})
    
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            old_status = o.get("delivery_status", "")
            o["delivery_status"] = status
            
            if status == "已收货":
                o["delivery_date"] = datetime.now().strftime("%Y-%m-%d")
                # 确认收货增加库存
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = p.get("quantity", 0) + o.get("quantity", 0)
                        break
            elif old_status == "已收货" and status == "未收货":
                # 取消收货减少库存
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = max(0, p.get("quantity", 0) - o.get("quantity", 0))
                        break
            break
    
    save('purchase', purchase_data)
    save('inventory', inv)
    return redirect(url_for('purchase.purchase'))


# ========== 编辑采购订单 ==========
@purchase_bp.route('/edit_purchase/<int:order_id>', methods=['GET', 'POST'])
def edit_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders": []})
    products = load('inventory', {"products": []}).get("products", [])
    
    if request.method == 'POST':
        product = request.form.get('product', '').strip()
        quantity = int(request.form.get('quantity', 0))
        unit_price = float(request.form.get('unit_price', 0))
        supplier = request.form.get('supplier', '').strip()
        
        total_amount = quantity * unit_price
        
        for o in purchase_data.get("orders", []):
            if o.get("id") == order_id:
                # 恢复原有库存
                old_quantity = o.get("quantity", 0)
                old_product = o.get("product", "")
                inv = load('inventory', {"products": []})
                for p in inv.get("products", []):
                    if p.get("name") == old_product:
                        p["quantity"] = p.get("quantity", 0) - old_quantity
                        break
                save('inventory', inv)
                
                # 更新订单
                o["product"] = product
                o["quantity"] = quantity
                o["unit_price"] = unit_price
                o["total_amount"] = total_amount
                o["supplier"] = supplier
                
                # 重新计算付款状态
                paid = o.get("paid_amount", 0)
                if paid >= total_amount:
                    o["payment_status"] = "已付款"
                elif paid > 0:
                    o["payment_status"] = "部分付款"
                else:
                    o["payment_status"] = "未付款"
                break
        
        save('purchase', purchase_data)
        
        # 扣减新库存（如果已收货）
        inv = load('inventory', {"products": []})
        for p in inv.get("products", []):
            if p.get("name") == product:
                p["quantity"] = p.get("quantity", 0) + quantity
                break
        save('inventory', inv)
        
        return redirect(url_for('purchase.purchase'))
    
    # GET: 显示编辑页面
    order = None
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='edit_purchase',
        order=order, products=products)


@purchase_bp.route('/delete_purchase/<int:order_id>')
def delete_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders": []})
    inv = load('inventory', {"products": []})
    
    # 找到订单并恢复库存
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            # 如果已收货，减少库存
            if o.get("delivery_status") == "已收货":
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = max(0, p.get("quantity", 0) - o.get("quantity", 0))
                        break
            break
    
    purchase_data["orders"] = [o for o in purchase_data.get("orders", []) if o.get("id") != order_id]
    save('purchase', purchase_data)
    save('inventory', inv)
    
    return redirect(url_for('purchase.purchase'))


# ========== 采购退货 ==========
@purchase_bp.route('/return_purchase/<int:order_id>', methods=['GET', 'POST'])
def return_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders": []})
    inv = load('inventory', {"products": []})
    
    if request.method == 'POST':
        return_quantity = int(request.form.get('return_quantity', 0))
        return_reason = request.form.get('reason', '').strip()
        refund_account = request.form.get('refund_account', '')
        refund_amount = float(request.form.get('refund_amount', 0))
        
        for o in purchase_data.get("orders", []):
            if o.get("id") == order_id:
                # 记录退货
                if "returns" not in o:
                    o["returns"] = []
                
                o["returns"].append({
                    "date": datetime.now().strftime("%Y-%m-%d"),
                    "quantity": return_quantity,
                    "reason": return_reason,
                    "refund_account": refund_account,
                    "refund_amount": refund_amount,
                    "operator": user.get("name", "")
                })
                
                # 减少库存
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = max(0, p.get("quantity", 0) - return_quantity)
                        break
                
                # 更新订单状态
                original_qty = o.get("quantity", 0)
                returned_qty = sum(r.get("quantity", 0) for r in o.get("returns", []))
                
                if returned_qty >= original_qty:
                    o["status"] = "已退货"
                else:
                    o["status"] = "部分退货"
                
                # 退款处理
                if refund_amount > 0:
                    o["paid_amount"] = max(0, o.get("paid_amount", 0) - refund_amount)
                    if o.get("paid_amount", 0) <= 0:
                        o["payment_status"] = "已退款"
                
                break
        
        save('purchase', purchase_data)
        save('inventory', inv)
        return redirect(url_for('purchase.purchase'))
    
    # GET: 显示退货页面
    order = None
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='return_purchase', order=order)


# ========== 取消采购订单 ==========
@purchase_bp.route('/cancel_purchase/<int:order_id>')
def cancel_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders": []})
    inv = load('inventory', {"products": []})
    
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            # 只有未付款+未收货可以取消
            if o.get("payment_status") != "已付款" and o.get("delivery_status") != "已收货":
                o["status"] = "已取消"
                # 如果已收货，需要减少库存
                if o.get("delivery_status") == "已收货":
                    for p in inv.get("products", []):
                        if p.get("name") == o.get("product"):
                            p["quantity"] = max(0, p.get("quantity", 0) - o.get("quantity", 0))
                            break
            break
    
    save('purchase', purchase_data)
    save('inventory', inv)
    return redirect(url_for('purchase.purchase'))
