# -*- coding: utf-8 -*-
"""葡月ERP - 主路由"""

from flask import Blueprint, render_template, request, redirect, url_for
from collections import defaultdict
from datetime import datetime
from ..utils import load, save, get_current_user, check_permission, render_with_user
from .auth import get_sessions

main_bp = Blueprint('main', __name__)


@main_bp.route('/')
def home():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get('username') == 'guest':
        return redirect(url_for('auth.login'))
    
    inv = load('inventory', {"products":[]})
    sales = load('sales', {"orders":[]})
    cust = load('customers', {"list":[]})
    purchase = load('purchase', {"orders":[]})
    expenses = load('expenses', {"list": []})
    
    orders = sales.get("orders", [])
    
    # 销售统计
    ts = sum(o.get("total_amount",0) for o in orders)
    tp = sum(o.get("profit",0) for o in orders)
    tq = sum(o.get("quantity",0) for o in orders)
    m = round(tp/ts*100,1) if ts else 0
    
    # 采购统计
    purchase_total = sum(o.get("total_amount", 0) for o in purchase.get("orders", []))
    purchase_paid = sum(o.get("total_amount", 0) for o in purchase.get("orders", []) if o.get("payment_status") == "已付款")
    
    # 费用统计
    expense_total = sum(e.get("amount", 0) for e in expenses.get("list", []) if e.get("status") == "已付款")
    
    # 库存统计
    inventory_value = sum(p.get("quantity", 0) * p.get("cost_price", 0) for p in inv.get("products", []))
    inventory_quantity = sum(p.get("quantity", 0) for p in inv.get("products", []))
    
    # 待处理事项
    pending_orders = len([o for o in orders if o.get("payment_status") != "已收款"])
    pending_purchase = len([o for o in purchase.get("orders", []) if o.get("payment_status") != "已付款"])
    pending_expenses = len([e for e in expenses.get("list", []) if e.get("status") == "待审批"])
    
    # 产品销量统计
    pd = defaultdict(lambda:{"q":0,"p":0})
    for o in orders:
        pd[o.get("product","")]["q"]+=o.get("quantity",0)
        pd[o.get("product","")]["p"]+=o.get("profit",0)
    tp_list = [{"name":k[:25],"quantity":v["q"],"profit":f"{v['p']:,.0f}"} for k,v in sorted(pd.items(),key=lambda x:x[1]["q"],reverse=True)[:5]]
    
    # 客户统计
    cd = defaultdict(lambda:{"q":0,"p":0})
    for o in orders:
        cd[o.get("customer","")]["q"]+=o.get("quantity",0)
        cd[o.get("customer","")]["p"]+=o.get("profit",0)
    tc_list = [{"name":k[:20],"quantity":v["q"],"profit":f"{v['p']:,.0f}"} for k,v in sorted(cd.items(),key=lambda x:x[1]["p"],reverse=True)[:5]]
    
    # 库存警告（含建议补货量）
    warn = []
    for p in inv.get("products", []):
        qty = p.get("quantity", 0)
        min_qty = p.get("min_stock", 0)
        if min_qty and qty < min_qty:
            warn.append({
                "name": p["name"],
                "quantity": qty,
                "min_stock": min_qty,
                "suggest": min_qty * 2 - qty  # 建议补货到安全库存的2倍
            })
    
    stats = {
        "total_sales": f"{ts:,.0f}",
        "total_profit": f"{tp:,.0f}",
        "total_orders": len(orders),
        "total_quantity": tq,
        "total_customers": len(cust.get("list",[])),
        "margin": m,
        "purchase_total": purchase_total,
        "purchase_paid": purchase_paid,
        "expense_total": expense_total,
        "inventory_value": inventory_value,
        "inventory_quantity": inventory_quantity,
        "pending_orders": pending_orders,
        "pending_purchase": pending_purchase,
        "pending_expenses": pending_expenses
    }
    
    return render_with_user('index.html', sessions, page='home', 
        stats=stats, top_products=tp_list, top_customers=tc_list, warnings=warn)


@main_bp.route('/sales')
def sales():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders":[]})
    products = load('inventory', {"products":[]}).get("products", [])
    customers = load('customers', {"list":[]}).get("list", [])
    orders = sales_data.get("orders", [])
    
    # 销售统计
    total_sales = sum(o.get("total_amount", 0) for o in orders)
    total_profit = sum(o.get("profit", 0) for o in orders)
    total_quantity = sum(o.get("quantity", 0) for o in orders)
    paid_amount = sum(o.get("total_amount", 0) for o in orders if o.get("payment_status") == "已收款")
    pending_amount = total_sales - paid_amount
    
    # 订单状态统计
    status_counts = {"已完成": 0, "处理中": 0, "已取消": 0}
    payment_status = {"已收款": 0, "待收款": 0}
    for o in orders:
        status = o.get("status", "已完成")
        if status in status_counts:
            status_counts[status] += 1
        else:
            status_counts["处理中"] += 1
        
        if o.get("payment_status") == "已收款":
            payment_status["已收款"] += 1
        else:
            payment_status["待收款"] += 1
    
    # 待收款订单
    unpaid_orders = [o for o in orders if o.get("payment_status") != "已收款"]
    partial_stats = {
        "count": len([o for o in orders if o.get("payment_status") == "部分收款"]),
        "total": sum(o.get("total_amount", 0) for o in orders if o.get("payment_status") == "部分收款"),
        "paid": sum(o.get("paid_amount", 0) for o in orders if o.get("payment_status") == "部分收款"),
        "remain": sum(o.get("total_amount", 0) - o.get("paid_amount", 0) for o in orders if o.get("payment_status") == "部分收款"),
        "overdue": len([o for o in unpaid_orders if o.get("payment_status") == "部分收款"])
    }
    
    sales_stats = {
        "total_sales": total_sales,
        "total_profit": total_profit,
        "total_quantity": total_quantity,
        "total_orders": len(orders),
        "paid_amount": paid_amount,
        "pending_amount": pending_amount,
        "status_counts": status_counts,
        "payment_status": payment_status
    }
    
    return render_with_user('index.html', sessions, page='sales',
        orders=orders,
        products=products,
        customers=customers,
        sales_stats=sales_stats,
        unpaid_orders=unpaid_orders,
        partial_stats=partial_stats)


@main_bp.route('/add_sale', methods=['POST'])
def add_sale():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders":[]})
    product = request.form.get('product', '').strip()
    quantity = int(request.form.get('quantity', 0))
    unit_price = float(request.form.get('unit_price', 0))
    customer = request.form.get('customer', '').strip()
    
    # 验证
    if not product or quantity <= 0 or unit_price < 0 or not customer:
        return redirect(url_for('main.sales'))
    
    # 检查库存
    products = load('inventory', {"products":[]}).get("products", [])
    cost_price = 0
    stock_quantity = 0
    for p in products:
        if p.get("name") == product:
            cost_price = float(p.get("cost_price", 0))
            stock_quantity = p.get("quantity", 0)
            break
    
    # 检查库存是否足够
    if stock_quantity < quantity:
        return redirect(url_for('main.sales'))
    
    total_amount = quantity * unit_price
    cost = quantity * cost_price
    profit = total_amount - cost
    
    # 使用最大ID+1，避免重复
    orders = sales_data.get("orders", [])
    max_id = max([o.get("id", 0) for o in orders]) if orders else 0
    
    order = {
        "id": max_id + 1,
        "product": product,
        "quantity": quantity,
        "unit_price": unit_price,
        "cost_price": cost_price,
        "total_amount": total_amount,
        "profit": profit,
        "customer": customer,
        "date": datetime.now().strftime("%Y-%m-%d"),
        "status": "已完成",
        "payment_status": "未收款",
        "delivery_status": "未发货",
        "paid_amount": 0
    }
    
    sales_data.setdefault("orders", []).append(order)
    save('sales', sales_data)
    
    # 更新库存
    inv = load('inventory', {"products":[]})
    for p in inv.get("products", []):
        if p.get("name") == product:
            p["quantity"] = p.get("quantity", 0) - quantity
            break
    save('inventory', inv)
    
    return redirect(url_for('main.sales'))


# ========== 编辑销售订单 ==========
@main_bp.route('/edit_sale/<int:order_id>', methods=['GET', 'POST'])
def edit_sale(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders": []})
    products = load('inventory', {"products": []}).get("products", [])
    customers = load('customers', {"list": []}).get("list", [])
    
    if request.method == 'POST':
        product = request.form.get('product', '').strip()
        quantity = int(request.form.get('quantity', 0))
        unit_price = float(request.form.get('unit_price', 0))
        customer = request.form.get('customer', '').strip()
        
        # 获取成本价
        cost_price = 0
        for p in products:
            if p.get("name") == product:
                cost_price = float(p.get("cost_price", 0))
                break
        
        total_amount = quantity * unit_price
        cost = quantity * cost_price
        profit = total_amount - cost
        
        for o in sales_data.get("orders", []):
            if o.get("id") == order_id:
                # 恢复原有库存
                old_quantity = o.get("quantity", 0)
                old_product = o.get("product", "")
                inv = load('inventory', {"products": []})
                for p in inv.get("products", []):
                    if p.get("name") == old_product:
                        p["quantity"] = p.get("quantity", 0) + old_quantity
                        break
                save('inventory', inv)
                
                # 更新订单
                o["product"] = product
                o["quantity"] = quantity
                o["unit_price"] = unit_price
                o["cost_price"] = cost_price
                o["total_amount"] = total_amount
                o["profit"] = profit
                o["customer"] = customer
                break
        
        save('sales', sales_data)
        
        # 扣减新库存
        inv = load('inventory', {"products": []})
        for p in inv.get("products", []):
            if p.get("name") == product:
                p["quantity"] = p.get("quantity", 0) - quantity
                break
        save('inventory', inv)
        
        return redirect(url_for('main.sales'))
    
    # GET: 显示编辑页面
    order = None
    for o in sales_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='edit_sale',
        order=order, products=products, customers=customers)


@main_bp.route('/delete_sale/<int:order_id>')
def delete_sale(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders": []})
    inv = load('inventory', {"products": []})
    
    # 找到订单并恢复库存
    for o in sales_data.get("orders", []):
        if o.get("id") == order_id:
            # 如果未发货，恢复库存
            if o.get("delivery_status") != "已发货":
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = p.get("quantity", 0) + o.get("quantity", 0)
                        break
            break
    
    sales_data["orders"] = [o for o in sales_data.get("orders", []) if o.get("id") != order_id]
    save('sales', sales_data)
    save('inventory', inv)
    
    return redirect(url_for('main.sales'))


# ========== 销售收款 ==========
@main_bp.route('/receive_payment/<int:order_id>', methods=['GET', 'POST'])
def receive_payment(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders": []})
    
    if request.method == 'POST':
        account = request.form.get('account', '')
        amount = float(request.form.get('amount', 0))
        
        for o in sales_data.get("orders", []):
            if o.get("id") == order_id:
                old_paid = o.get("paid_amount", 0)
                new_paid = old_paid + amount
                total = o.get("total_amount", 0)
                
                if new_paid >= total:
                    o["payment_status"] = "已收款"
                elif new_paid > 0:
                    o["payment_status"] = "部分收款"
                else:
                    o["payment_status"] = "未收款"
                
                if "payment_records" not in o:
                    o["payment_records"] = []
                
                o["payment_records"].append({
                    "date": datetime.now().strftime("%Y-%m-%d %H:%M"),
                    "account": account,
                    "amount": amount,
                    "operator": user.get("name", "")
                })
                
                o["paid_amount"] = new_paid
                o["payment_date"] = datetime.now().strftime("%Y-%m-%d")
                break
        
        save('sales', sales_data)
        return redirect(url_for('main.sales'))
    
    order = None
    for o in sales_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='receive_payment', order=order)


# ========== 快速收款 ==========
@main_bp.route('/quick_receive/<int:order_id>/<account>')
def quick_receive(order_id, account):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders": []})
    
    for o in sales_data.get("orders", []):
        if o.get("id") == order_id:
            total = o.get("total_amount", 0)
            old_paid = o.get("paid_amount", 0)
            amount = total - old_paid
            
            if amount > 0:
                o["payment_status"] = "已收款"
                
                if "payment_records" not in o:
                    o["payment_records"] = []
                
                o["payment_records"].append({
                    "date": datetime.now().strftime("%Y-%m-%d %H:%M"),
                    "account": account,
                    "amount": amount,
                    "operator": user.get("name", "")
                })
                
                o["paid_amount"] = total
                o["payment_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('sales', sales_data)
    return redirect(url_for('main.sales'))


# ========== 录单页面 ==========
@main_bp.route('/add_sale_page')
def add_sale_page():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    products = load('inventory', {"products":[]}).get("products", [])
    customers = load('customers', {"list":[]}).get("list", [])
    
    return render_with_user('index.html', sessions, page='add_sale',
        products=products, customers=customers)


# ========== 分析页面 ==========
@main_bp.route('/analysis')
def analysis():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders":[]})
    orders = sales.get("orders", [])
    
    monthly = defaultdict(lambda: {"sales": 0, "profit": 0, "quantity": 0})
    for o in orders:
        month = o.get("date", "")[:7]
        monthly[month]["sales"] += o.get("total_amount", 0)
        monthly[month]["profit"] += o.get("profit", 0)
        monthly[month]["quantity"] += o.get("quantity", 0)
    
    monthly_list = [{"month": k, "sales": v["sales"], "profit": v["profit"], "quantity": v["quantity"]} 
                    for k, v in sorted(monthly.items())]
    
    return render_with_user('index.html', sessions, page='analysis', monthly_data=monthly_list)


@main_bp.route('/product_analysis')
def product_analysis():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders":[]})
    orders = sales.get("orders", [])
    
    products = defaultdict(lambda: {"quantity": 0, "sales": 0, "profit": 0})
    for o in orders:
        pname = o.get("product", "")
        products[pname]["quantity"] += o.get("quantity", 0)
        products[pname]["sales"] += o.get("total_amount", 0)
        products[pname]["profit"] += o.get("profit", 0)
    
    product_list = [{"name": k, "quantity": v["quantity"], "sales": v["sales"], "profit": v["profit"]}
                    for k, v in sorted(products.items(), key=lambda x: x[1]["sales"], reverse=True)]
    
    product_stats_summary = {
        "total_products": len(product_list),
        "total_sales": sum(p["sales"] for p in product_list),
        "total_profit": sum(p["profit"] for p in product_list)
    }
    
    return render_with_user('index.html', sessions, page='product_analysis', 
        products=product_list,
        product_stats_summary=product_stats_summary)


@main_bp.route('/customer_analysis')
def customer_analysis():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders":[]})
    orders = sales.get("orders", [])
    
    customers = defaultdict(lambda: {"quantity": 0, "sales": 0, "orders": 0})
    for o in orders:
        cname = o.get("customer", "")
        customers[cname]["quantity"] += o.get("quantity", 0)
        customers[cname]["sales"] += o.get("total_amount", 0)
        customers[cname]["orders"] += 1
    
    customer_list = [{"name": k, "quantity": v["quantity"], "sales": v["sales"], "orders": v["orders"]}
                      for k, v in sorted(customers.items(), key=lambda x: x[1]["sales"], reverse=True)]
    
    customer_stats_summary = {
        "total_customers": len(customer_list),
        "total_sales": sum(c["sales"] for c in customer_list),
        "total_orders": sum(c["orders"] for c in customer_list)
    }
    
    customer_stats = {
        "total": len(customer_list),
        "active": len([c for c in customer_list if c["orders"] > 1]),
        "inactive": len([c for c in customer_list if c["orders"] == 1]),
        "repeat_rate": round(len([c for c in customer_list if c["orders"] > 1]) / len(customer_list) * 100, 1) if customer_list else 0
    }
    
    # 客户等级分布 - 从客户数据获取
    cust_data = load('customers', {"list": []})
    level_distribution = {}
    for c in cust_data.get("list", []):
        level = c.get("level", "普通")
        level_distribution[level] = level_distribution.get(level, 0) + 1
    
    return render_with_user('index.html', sessions, page='customer_analysis', 
        customers=customer_list,
        customer_stats_summary=customer_stats_summary,
        customer_stats=customer_stats,
        level_distribution=level_distribution)


@main_bp.route('/customer_orders/<customer_name>')
def customer_orders(customer_name):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders":[]})
    orders = [o for o in sales.get("orders", []) if o.get("customer") == customer_name]
    
    return render_with_user('index.html', sessions, page='customer_orders', 
                          customer_name=customer_name, orders=orders)


@main_bp.route('/confirm_payment/<int:order_id>/<status>')
def confirm_payment(order_id, status):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders":[]})
    for o in sales.get("orders", []):
        if o.get("id") == order_id:
            if status == "未收款":
                # 取消收款时，清空已收金额
                o["payment_status"] = status
                o["paid_amount"] = 0
                if "payment_records" in o:
                    o["payment_records"] = []
            else:
                o["payment_status"] = status
            break
    save('sales', sales)
    return redirect(url_for('main.sales'))


@main_bp.route('/confirm_delivery/<int:order_id>/<status>')
def confirm_delivery(order_id, status):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders":[]})
    inv = load('inventory', {"products":[]})
    
    for o in sales.get("orders", []):
        if o.get("id") == order_id:
            old_status = o.get("delivery_status", "")
            o["delivery_status"] = status
            
            # 处理库存
            if old_status == "未发货" and status == "已发货":
                # 发货扣减库存
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = p.get("quantity", 0) - o.get("quantity", 0)
                        break
            elif old_status == "已发货" and status == "未发货":
                # 取消发货恢复库存
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = p.get("quantity", 0) + o.get("quantity", 0)
                        break
            break
    
    save('sales', sales)
    save('inventory', inv)
    return redirect(url_for('main.sales'))
    return redirect(url_for('main.sales'))


@main_bp.route('/delete_order/<int:order_id>')
def delete_order(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales = load('sales', {"orders":[]})
    sales["orders"] = [o for o in sales.get("orders", []) if o.get("id") != order_id]
    save('sales', sales)
    return redirect(url_for('main.sales'))


# ========== 销售退货 ==========
@main_bp.route('/return_sale/<int:order_id>', methods=['GET', 'POST'])
def return_sale(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders": []})
    inv = load('inventory', {"products": []})
    
    if request.method == 'POST':
        return_quantity = int(request.form.get('return_quantity', 0))
        return_reason = request.form.get('reason', '').strip()
        refund_account = request.form.get('refund_account', '')
        refund_amount = float(request.form.get('refund_amount', 0))
        
        for o in sales_data.get("orders", []):
            if o.get("id") == order_id:
                # 记录退货
                if "returns" not in o:
                    o["returns"] = []
                
                o["returns"].append({
                    "date": datetime.now().strftime("%Y-%m-%d"),
                    "quantity": return_quantity,
                    "reason": return_reason,
                    "refund_account": refund_account,
                    "refund_amount": refund_amount,
                    "operator": user.get("name", "")
                })
                
                # 恢复库存
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = p.get("quantity", 0) + return_quantity
                        break
                
                # 更新订单状态
                original_qty = o.get("quantity", 0)
                returned_qty = sum(r.get("quantity", 0) for r in o.get("returns", []))
                
                if returned_qty >= original_qty:
                    o["status"] = "已退货"
                    o["payment_status"] = "已退款"
                else:
                    o["status"] = "部分退货"
                
                # 退款处理
                if refund_amount > 0:
                    o["paid_amount"] = max(0, o.get("paid_amount", 0) - refund_amount)
                    if o.get("paid_amount", 0) <= 0:
                        o["payment_status"] = "已退款"
                
                break
        
        save('sales', sales_data)
        save('inventory', inv)
        return redirect(url_for('main.sales'))
    
    # GET: 显示退货页面
    order = None
    for o in sales_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='return_sale', order=order)


# ========== 取消销售订单 ==========
@main_bp.route('/cancel_sale/<int:order_id>')
def cancel_sale(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    sales_data = load('sales', {"orders": []})
    inv = load('inventory', {"products": []})
    
    for o in sales_data.get("orders", []):
        if o.get("id") == order_id:
            # 只有未收款+未发货可以取消
            if o.get("payment_status") != "已收款" and o.get("delivery_status") != "已发货":
                o["status"] = "已取消"
                # 恢复库存
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = p.get("quantity", 0) + o.get("quantity", 0)
                        break
            break
    
    save('sales', sales_data)
    save('inventory', inv)
    return redirect(url_for('main.sales'))
