# -*- coding: utf-8 -*-
"""葡月ERP - 备份管理路由"""

from flask import Blueprint, render_template, request, redirect, url_for, send_file, current_app
from datetime import datetime
import os
import json
import zipfile
import shutil
import smtplib
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from email.mime.base import MIMEBase
from email import encoders
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

backup_bp = Blueprint('backup', __name__)

# 邮件配置
SENDER_EMAIL = "19725200@qq.com"
SENDER_PASSWORD = "uxczdvtcqqtbdj"  # QQ邮箱授权码
RECIPIENT_EMAIL = "gustyji@gmail.com"
SMTP_SERVER = "smtp.qq.com"
SMTP_PORT = 587


def send_backup_email(backup_path, subject, description):
    """发送备份邮件"""
    try:
        msg = MIMEMultipart()
        msg['From'] = SENDER_EMAIL
        msg['To'] = RECIPIENT_EMAIL
        msg['Subject'] = subject
        
        body = f"""
🍷 葡月ERP系统2026 备份通知

{description}

备份时间: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}

此邮件由系统自动发送。
        """
        msg.attach(MIMEText(body, 'plain', 'utf-8'))
        
        # 添加附件
        if backup_path and os.path.exists(backup_path):
            with open(backup_path, 'rb') as f:
                part = MIMEBase('application', 'octet-stream')
                part.set_payload(f.read())
            encoders.encode_base64(part)
            part.add_header('Content-Disposition', f'attachment; filename= {os.path.basename(backup_path)}')
            msg.attach(part)
        
        server = smtplib.SMTP(SMTP_SERVER, SMTP_PORT)
        server.starttls()
        server.login(SENDER_EMAIL, SENDER_PASSWORD)
        server.sendmail(SENDER_EMAIL, RECIPIENT_EMAIL, msg.as_string())
        server.quit()
        print(f"✅ 邮件发送成功: {subject}")
        return True
    except Exception as e:
        print(f"❌ 邮件发送失败: {e}")
        return False


@backup_bp.route('/backup')
def backup():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    if user.get("role") != "admin":
        return "无权限访问", 403
    
    # 获取备份列表
    backup_dir = current_app.config['BACKUP_DIR'] if hasattr(backup_bp, 'current_app') else "/root/data/disk/clawde/workspace/users/wecom_GuSi/backups"
    
    backups = []
    if os.path.exists(backup_dir):
        for f in os.listdir(backup_dir):
            if f.endswith('.zip'):
                path = os.path.join(backup_dir, f)
                stat = os.stat(path)
                backups.append({
                    "name": f,
                    "size": round(stat.st_size / 1024 / 1024, 2),
                    "date": datetime.fromtimestamp(stat.st_mtime).strftime("%Y-%m-%d %H:%M")
                })
    
    backups.sort(key=lambda x: x["date"], reverse=True)
    
    return render_with_user('index.html', sessions, page='backup', backups=backups)


@backup_bp.route('/do_backup')
def do_backup():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get("role") != "admin":
        return "无权限访问", 403
    
    data_dir = "/root/data/disk/clawde/workspace/users/wecom_GuSi/data"
    backup_dir = "/root/data/disk/clawde/workspace/users/wecom_GuSi/backups"
    os.makedirs(backup_dir, exist_ok=True)
    
    # 创建备份文件名
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    backup_file = os.path.join(backup_dir, f"backup_{timestamp}.zip")
    
    # 创建压缩包
    with zipfile.ZipFile(backup_file, 'w', zipfile.ZIP_DEFLATED) as zf:
        for f in os.listdir(data_dir):
            file_path = os.path.join(data_dir, f)
            if os.path.isfile(file_path):
                zf.write(file_path, f)
    
    # 发送邮件通知
    send_backup_email(backup_file, f"🍷 葡月ERP系统2026 数据备份 - {datetime.now().strftime('%Y-%m-%d %H:%M')}", "数据备份已完成")
    
    return redirect(url_for('backup.backup'))


@backup_bp.route('/do_system_backup')
def do_system_backup():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get("role") != "admin":
        return "无权限访问", 403
    
    data_dir = "/root/data/disk/clawde/workspace/users/wecom_GuSi"
    backup_dir = "/root/data/disk/clawde/workspace/users/wecom_GuSi/backups"
    os.makedirs(backup_dir, exist_ok=True)
    
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    backup_file = os.path.join(backup_dir, f"system_{timestamp}.zip")
    
    with zipfile.ZipFile(backup_file, 'w', zipfile.ZIP_DEFLATED) as zf:
        for root, dirs, files in os.walk(data_dir):
            for f in files:
                if not f.endswith('.zip'):
                    file_path = os.path.join(root, f)
                    arcname = os.path.relpath(file_path, data_dir)
                    zf.write(file_path, arcname)
    
    # 发送邮件通知
    send_backup_email(backup_file, f"🍷 葡月ERP系统2026 系统备份 - {datetime.now().strftime('%Y-%m-%d %H:%M')}", "系统备份文件已生成")
    
    return redirect(url_for('backup.backup'))


@backup_bp.route('/download_backup/<name>')
def download_backup(name):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user or user.get("role") != "admin":
        return "无权限访问", 403
    
    backup_dir = "/root/data/disk/clawde/workspace/users/wecom_GuSi/backups"
    backup_file = os.path.join(backup_dir, name)
    
    if os.path.exists(backup_file):
        return send_file(backup_file, as_attachment=True)
    
    return "备份文件不存在", 404
