# -*- coding: utf-8 -*-
"""葡月ERP - 采购路由"""

from flask import Blueprint, render_template, request, redirect, url_for
from datetime import datetime
from ..utils import load, save, get_current_user, render_with_user
from .auth import get_sessions

purchase_bp = Blueprint('purchase', __name__)


@purchase_bp.route('/purchase')
def purchase():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    products = load('inventory', {"products":[]}).get("products", [])
    orders = purchase_data.get("orders", [])
    
    # 采购统计
    total = sum(o.get("total_amount", 0) for o in orders)
    paid = sum(o.get("total_amount", 0) for o in orders if o.get("payment_status") == "已付款")
    pending = total - paid
    
    # 订单状态
    payment_status = {"已付款": 0, "待付款": 0}
    delivery_status = {"已收货": 0, "待收货": 0}
    for o in orders:
        if o.get("payment_status") == "已付款":
            payment_status["已付款"] += 1
        else:
            payment_status["待付款"] += 1
        
        if o.get("delivery_status") == "已收货":
            delivery_status["已收货"] += 1
        else:
            delivery_status["待收货"] += 1
    
    purchase_stats = {
        "total": total,
        "paid": paid,
        "pending": pending,
        "orders": len(orders),
        "quantity": sum(o.get("quantity", 0) for o in orders),
        "payment_status": payment_status,
        "delivery_status": delivery_status
    }
    
    return render_with_user('index.html', sessions, page='purchase',
        orders=orders,
        products=products,
        purchase_stats=purchase_stats)


@purchase_bp.route('/add_purchase', methods=['POST'])
def add_purchase():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    
    product = request.form.get('product', '').strip()
    quantity = int(request.form.get('quantity', 0))
    unit_price = float(request.form.get('unit_price', 0))
    supplier = request.form.get('supplier', '').strip()
    
    total_amount = quantity * unit_price
    
    order = {
        "id": len(purchase_data.get("orders",[])) + 1,
        "product": product,
        "quantity": quantity,
        "unit_price": unit_price,
        "total_amount": total_amount,
        "supplier": supplier,
        "date": datetime.now().strftime("%Y-%m-%d"),
        "status": "待付款",
        "payment_status": "未付款",
        "delivery_status": "未发货"
    }
    
    purchase_data.setdefault("orders", []).append(order)
    save('purchase', purchase_data)
    
    return redirect(url_for('purchase.purchase'))


@purchase_bp.route('/confirm_purchase_payment/<int:order_id>/<status>')
@purchase_bp.route('/confirm_purchase_payment/<int:order_id>/<status>/<account>')
@purchase_bp.route('/confirm_purchase_payment/<int:order_id>/<status>/<account>/<amount>')
def confirm_purchase_payment(order_id, status, account=None, amount=None):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            o["payment_status"] = status
            if account:
                o["payment_account"] = account
            if amount:
                o["paid_amount"] = float(amount)
            if status == "已付款":
                o["payment_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('purchase', purchase_data)
    return redirect(url_for('purchase.purchase'))


@purchase_bp.route('/pay_purchase/<int:order_id>')
def pay_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    products = load('inventory', {"products":[]}).get("products", [])
    
    order = None
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='pay_purchase',
        order=order, products=products)


@purchase_bp.route('/do_pay_purchase', methods=['POST'])
def do_pay_purchase():
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    order_id = int(request.form.get('order_id', 0))
    account = request.form.get('account', '')
    amount = float(request.form.get('amount', 0))
    
    purchase_data = load('purchase', {"orders":[]})
    
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            # 获取之前的已付金额
            old_paid = o.get("paid_amount", 0)
            new_paid = old_paid + amount
            total = o.get("total_amount", 0)
            
            # 更新付款状态
            if new_paid >= total:
                o["payment_status"] = "已付款"
            elif new_paid > 0:
                o["payment_status"] = "部分付款"
            else:
                o["payment_status"] = "未付款"
            
            # 记录付款信息
            if "payment_records" not in o:
                o["payment_records"] = []
            
            o["payment_records"].append({
                "date": datetime.now().strftime("%Y-%m-%d %H:%M"),
                "account": account,
                "amount": amount,
                "operator": user.get("name", "")
            })
            
            o["paid_amount"] = new_paid
            o["payment_date"] = datetime.now().strftime("%Y-%m-%d")
            break
    
    save('purchase', purchase_data)
    return redirect(url_for('purchase.purchase'))


@purchase_bp.route('/confirm_purchase_delivery/<int:order_id>/<status>')
def confirm_purchase_delivery(order_id, status):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            o["delivery_status"] = status
            if status == "已收货":
                o["delivery_date"] = datetime.now().strftime("%Y-%m-%d")
                # 更新库存
                inv = load('inventory', {"products":[]})
                for p in inv.get("products", []):
                    if p.get("name") == o.get("product"):
                        p["quantity"] = p.get("quantity", 0) + o.get("quantity", 0)
                        break
                save('inventory', inv)
            break
    
    save('purchase', purchase_data)
    return redirect(url_for('purchase.purchase'))


# ========== 编辑采购订单 ==========
@purchase_bp.route('/edit_purchase/<int:order_id>', methods=['GET', 'POST'])
def edit_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders": []})
    products = load('inventory', {"products": []}).get("products", [])
    
    if request.method == 'POST':
        product = request.form.get('product', '').strip()
        quantity = int(request.form.get('quantity', 0))
        unit_price = float(request.form.get('unit_price', 0))
        supplier = request.form.get('supplier', '').strip()
        
        total_amount = quantity * unit_price
        
        for o in purchase_data.get("orders", []):
            if o.get("id") == order_id:
                # 恢复原有库存
                old_quantity = o.get("quantity", 0)
                old_product = o.get("product", "")
                inv = load('inventory', {"products": []})
                for p in inv.get("products", []):
                    if p.get("name") == old_product:
                        p["quantity"] = p.get("quantity", 0) - old_quantity
                        break
                save('inventory', inv)
                
                # 更新订单
                o["product"] = product
                o["quantity"] = quantity
                o["unit_price"] = unit_price
                o["total_amount"] = total_amount
                o["supplier"] = supplier
                
                # 重新计算付款状态
                paid = o.get("paid_amount", 0)
                if paid >= total_amount:
                    o["payment_status"] = "已付款"
                elif paid > 0:
                    o["payment_status"] = "部分付款"
                else:
                    o["payment_status"] = "未付款"
                break
        
        save('purchase', purchase_data)
        
        # 扣减新库存（如果已收货）
        inv = load('inventory', {"products": []})
        for p in inv.get("products", []):
            if p.get("name") == product:
                p["quantity"] = p.get("quantity", 0) + quantity
                break
        save('inventory', inv)
        
        return redirect(url_for('purchase.purchase'))
    
    # GET: 显示编辑页面
    order = None
    for o in purchase_data.get("orders", []):
        if o.get("id") == order_id:
            order = o
            break
    
    return render_with_user('index.html', sessions, page='edit_purchase',
        order=order, products=products)


@purchase_bp.route('/delete_purchase/<int:order_id>')
def delete_purchase(order_id):
    sessions = get_sessions()
    user = get_current_user(sessions)
    if not user:
        return redirect(url_for('auth.login'))
    
    purchase_data = load('purchase', {"orders":[]})
    purchase_data["orders"] = [o for o in purchase_data.get("orders", []) if o.get("id") != order_id]
    save('purchase', purchase_data)
    
    return redirect(url_for('purchase.purchase'))
